//************************************************************
// Header file Change.h
//************************************************************
// This file contains three functions. They can translate 
// .dat format to PV, Xmol XYZ and Tinker XYZ format.
// Their names are:
//     ChangePV()
//     ChangeTinker()
//     ChangeXmol()
//************************************************************
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

//************************************************************
// Function ChangePV()
//************************************************************
// This function changes from .dat file to PV file.
// 
// *** Arguments *** 
//   fninp[]: input filename.
//   fnoup[]: output filename.
//   FDEBUG: flag (zero: true, non-zero:false)
//        If true, this function displays the debug information.
//   fcenter: flag (zero: true, non-zero:false)
//      fcenter represents the origin in input file.
//      If true,  origin is set to the center of the cell.
//      IF false, origin is set to the corner of the cell.
//************************************************************
int ChangePV(char fninp[], char fnoup[], int FDEBUG, int fcenter)
{
    //******************************
    // define struct variables
    //******************************
    #pragma pack(1)
    struct Header{ 
        char String[32];
        unsigned short mtype,nmol,nbond,ndat;
        double vlx1,vlx2,vly1,vly2,vlz1,vlz2;
        double time0,dt;
    } h;
    #pragma pack()
    
    #pragma pack(1)
    struct Atom{
        unsigned char Attr;
        short x,y,z;
    } *AtomBuf;
    #pragma pack()
    
    #pragma pack(1)
    struct Bond{
        unsigned char Attr;
        unsigned short p1,p2;
    } *BondBuf;
    #pragma pack()
    

    //******************************
    // define consts and variables
    //******************************
    FILE *fo,*fp;
    const double LBond = 1.8e0;
    const double BallMag = 0.8;
    unsigned char *AtomPot;
    
    int i,j,ibond,istep;
    double r, x, y, z;
    double vlx, vly, vlz;
    
    //******************************
    // Start function
    //******************************
    if (FDEBUG) {
        printf ("\n");
        printf ("Making pv file...\n");
        printf (" -- input file:  %s\n", fninp);
        printf (" -- output file: %s\n", fnoup);
    }
    
    //******************************
    // open files
    //******************************
    fp = fopen(fninp,"r");
    fo = fopen(fnoup,"wb");
    if (fp==NULL || ferror(fp)){
        printf("Error: file access failed for %s\n",fninp);
        return(EXIT_FAILURE);
    }
    
    //******************************
    // Reading and writing header
    //******************************
    if (FDEBUG) printf ("Writing Header...\n");
    strncpy(h.String,"PV-01 /Shoji-Maruyama Laboratory",32);
    fscanf(fp,"%hu %hu %hu", &(h.mtype), &(h.nmol), &(h.ndat));
    fscanf(fp,"%lf %lf %lf",&vlx, &vly, &vlz);
    fscanf(fp,"%lf %lf", &(h.time0), &(h.dt));
    
    h.nbond    = h.nmol*2;
    h.vlx1 = -vlx/2.0;  h.vly1 = -vly/2.0;   h.vlz1 = -vlz/2.0;
    h.vlx2 =  vlx/2.0;  h.vly2 =  vly/2.0;   h.vlz2 =  vlz/2.0;
    fwrite(&h, sizeof(struct Header), 1, fo);
    
    //******************************
    // Writing Ball-mag
    //******************************
    if (FDEBUG) printf ("Writing Ball-mag (=%lf)...\n",BallMag);
    x = BallMag;
    for(i=0;i<h.nmol;i++) {
        fwrite (&x, sizeof(double), 1, fo);
    }
    //******************************
    // Reading and writing every scene
    //******************************
    if (FDEBUG) printf ("Writing every scene...\n");
    AtomBuf = (struct Atom *)malloc(h.nmol * sizeof(struct Atom));
    BondBuf = (struct Bond *)malloc(h.nbond * sizeof(struct Bond));
    AtomPot = (unsigned char *)malloc(h.nmol * sizeof(unsigned char));
    
    for(istep=1;istep<=(int)h.ndat;istep++) {
        
        if (FDEBUG && istep%10==0)
            printf (" -- step:%4d\n",istep);
        
        //------------------------------
        // Filling AtomBuf...
        //------------------------------
        for(i=0;i<h.nmol;i++){
            fscanf(fp,"%lf %lf %lf",&x,&y,&z);
            if (!fcenter) {
                x -= vlx/2.0;
                y -= vlx/2.0;
                z -= vlx/2.0;
            }
            AtomBuf[i].Attr = (unsigned char)0;
            AtomBuf[i].x = (short)floor(x*1.0e2+0.5);
            AtomBuf[i].y = (short)floor(y*1.0e2+0.5);
            AtomBuf[i].z = (short)floor(z*1.0e2+0.5);
        }
        
        //------------------------------
        // Filling BondBuf...
        //------------------------------
        for(i=0;i<h.nmol;i++) AtomPot[i] = 0;
        
        ibond = 0;
        for(i=0;i<h.nmol-1;i++){
            for(j=i+1;j<h.nmol;j++){
                x = ((double)AtomBuf[i].x - (double)AtomBuf[j].x)/1.0e2;
                y = ((double)AtomBuf[i].y - (double)AtomBuf[j].y)/1.0e2;
                z = ((double)AtomBuf[i].z - (double)AtomBuf[j].z)/1.0e2;
                r = sqrt(x*x + y*y + z*z);
                
                if(FDEBUG && r<1.0) {
                    printf(" -- atom %d and %d are too close",i,j);
                    printf("(bond length=%lf)\n",r);
                }
                if (r>=LBond) continue;
                
                AtomPot[i]++;
                AtomPot[j]++;
                if(ibond<h.nbond){
                    BondBuf[ibond].Attr = (unsigned char)(0x76);
                    BondBuf[ibond].p1 = (unsigned short)(i+1);
                    BondBuf[ibond].p2 = (unsigned short)(j+1);
                    ibond++;
                }
            }
        }
        
        if(ibond<=h.nbond){
            for(i=ibond;i<h.nbond;i++) BondBuf[i].Attr=0xff;
            for(i=0;i<h.nmol;i++){
                switch(AtomPot[i]){
                    case 0:  AtomBuf[i].Attr = 15; break;
                    case 1:  AtomBuf[i].Attr = 14; break;
                    case 2:  AtomBuf[i].Attr = 10; break;
                    case 3:  AtomBuf[i].Attr = 2;  break;
                    default: AtomBuf[i].Attr = 1;  break;
                }
                if (AtomBuf[i].x==0 && AtomBuf[i].y==0 && AtomBuf[i].z==0)
                    AtomBuf[i].Attr = 3;
            }
        } else{
            printf("Bond number exceeded the limit. (ibond=%d\n",ibond);
            exit(EXIT_FAILURE);
        }
        fwrite(AtomBuf, sizeof(struct Atom), h.nmol, fo);
        fwrite(BondBuf, sizeof(struct Bond), h.nbond, fo);
    }
    free(AtomPot);
    free((void *)AtomBuf);
    free((void *)BondBuf);

    fclose(fp);
    fclose(fo);

    return(EXIT_SUCCESS);
}










//************************************************************
// Function ChangeTinker()
//************************************************************
// This function changes from .dat file to Tinker XYZ file.
// 
// *** Arguments *** 
//   fninp[]: input filename.
//   fnoup[]: output filename.
//   FDEBUG: flag (zero: true, non-zero:false)
//        If true, this function displays the debug information.
//   desc[]: description for the second line.
// 
// ================================================
//     About the format for the Tinker XYZ file
// ================================================
//
// The TINKER .xyz file contains a title line followed
// by one line for each atom in the structure. 
// Each line contains: the sequential number within the structure, 
// an atomic symbol or name, X-, Y-, and Z-coordinates, 
// the force field atom type number of the atom, 
// and a list of the atoms connected to the current atom. 
// Except for programs whose basic operation is in torsional space,
// all TINKER calculations are done from some version of the .xyz format.
// 
// Below is an example of TINKER .xyz format made by MOLDA.
// ------------------------------
//     17  TINKER xyz by MOLDA
//      1  C     -2.514810    0.059877    0.308041     1     2     6     7     8
//      2  C     -1.257405   -0.109775   -0.564742     1     1     3     9    10
//      3  C      0.000000    0.059877    0.308041     1     2     4    11    12
//      4  C      1.257405   -0.109775   -0.564742     1     3     5    13    14
//      5  C      2.514809    0.059877    0.308041     1     4    15    16    17
//      6  H     -3.404791   -0.060201   -0.309708     5     1
//      7  H     -2.514810    1.053585    0.755974     5     1
//      8  H     -2.514810   -0.693674    1.095607     5     1
//      9  H     -1.257405    0.643776   -1.352308     5     2
//     10  H     -1.257405   -1.103483   -1.012676     5     2
//     11  H      0.000000    1.053585    0.755974     5     3
//     12  H      0.000000   -0.693674    1.095607     5     3
//     13  H      1.257405    0.643776   -1.352308     5     4
//     14  H      1.257405   -1.103483   -1.012676     5     4
//     15  H      2.514809    1.053585    0.755974     5     5
//     16  H      2.514809   -0.693674    1.095607     5     5
//     17  H      3.404790   -0.060201   -0.309708     5     5
// ------------------------------
//
//************************************************************

int ChangeTinker(char fninp[], char fnoup[], int FDEBUG, char desc[])
{
    struct xyz{
        double x;
        double y;
        double z;
    } *q;
    //******************************
    // define consts and variables
    //******************************
    FILE *fo, *fp;
    const double LBond = 1.8e0;
    char fntmp[200];
    char *c;
    
    int nmol, igomi, ndat;
    double dgomi;
    int i,j,istep;
    double r;
    double x,y,z;
    double vlx, vly, vlz;
    if (FDEBUG) printf ("Making tinker XYZ file...\n");
    
    //******************************
    // Reading header
    //******************************
    fp = fopen(fninp,"r");
    if (fp==NULL || ferror(fp)){
        printf("Error: file access failed for %s\n",fninp);
        return(EXIT_FAILURE);
    }
    fscanf(fp,"%d %d %d", &igomi, &nmol, &ndat);
    fscanf(fp,"%lf %lf %lf",&vlx,&vly,&vlz);
    fscanf(fp,"%lf %lf", &dgomi, &dgomi);
    if (FDEBUG) {
        printf (" -- nmol: %d\n",nmol);
        printf (" -- ndat: %d\n",ndat);
    }
    
    q = (struct xyz*)malloc(nmol * sizeof(struct xyz));
    for(istep=1;istep<=ndat;istep++)
    {
        if (FDEBUG && istep%10==0)
            printf (" -- step:%4d\n",istep);
        
        //******************************
        // open output file
        //******************************
        strcpy(fntmp, fnoup);
        c = strrchr(fntmp, (int)'.');
        if (*c==(char)NULL) c = &(fntmp[strlen(fnoup)]);
        sprintf (c, "%03d.xyz", istep);
        fo = fopen(fntmp,"w");
        
        //******************************
        // Reading and writing header
        //******************************
        fprintf(fo,"%5d  %s\n", nmol, desc);
        
        //******************************
        // Reading and writing every scene
        //******************************
        for(i=0;i<nmol;i++) {
            fscanf(fp,"%lf %lf %lf", &(q[i].x), &(q[i].y), &(q[i].z));
        }
        
        for(i=0;i<nmol;i++){
            fprintf(fo, "%6d  ",i+1);
            fprintf(fo, "C   ");
            fprintf(fo, "%11.6lf ", q[i].x);
            fprintf(fo, "%11.6lf ", q[i].y);
            fprintf(fo, "%11.6lf ", q[i].z);
            fprintf(fo, "%5d ",1);
            for (j=0; j<nmol; j++) {
                if (i==j) continue;
                x = q[i].x - q[j].x;
                y = q[i].y - q[j].y;
                z = q[i].z - q[j].z;
                r = sqrt(x*x + y*y + z*z);
                if (r>=LBond) continue;
                fprintf (fo,"%5d ",j+1);
            }
            fprintf(fo, "\n");
        }
        fclose(fo);
    }
    free(q);
    fclose(fp);
    return(EXIT_SUCCESS);
}










//************************************************************
// Function ChangeXmol()
//************************************************************
// This function changes from .dat file to Xmol XYZ file.
// 
// *** Arguments *** 
//   fninp[]: input filename.
//   fnoup[]: output filename.
//   FDEBUG: flag (zero: true, non-zero:false)
//        If true, this function displays the debug information.
//   desc[]: description for the second line.
// 
// ==============================================
//     About the format for the Xmol XYZ file
// ==============================================
//
// The format for the XMol XYZ file format is quite simple.
// 
// The first line contains the number of atoms in the molecule.
// The second line contains the name or title of the molecule.
// It can be blank if desired.
// The rest of the file is composed of one line for each atom
// in the molecule. Each atom is designated by the atomic number
// and then the x, y, and z coordinate value.
// Separate each value by at least 1 space;
// there is no strict format for these lines. 
// 
// As an example, below is the entire file for toluene: 
// ------------------------------
// 15
// Toluene form the Cambridge Crystallographic Database
//   C        1.58890       -1.44870       -0.47000
//   C        1.54300       -2.25990        0.77910
//   C        2.21440       -3.47410        0.88040
//   C        2.16940       -4.22350        2.03080
//   C        1.45120       -3.77740        3.11890
//   C        0.77320       -2.58240        3.03840
//   C        0.82440       -1.82660        1.89060
//   H        2.40210       -1.69480       -0.94010
//   H        0.81370       -1.61080       -1.10630
//   H        1.55340       -0.51550       -0.20780
//   H        2.79760       -3.75410        0.13240
//   H        2.62530       -5.03180        2.07500
//   H        1.44160       -4.26200        3.93440
//   H        0.18740       -2.32460        3.76300
//   H        0.34860       -0.96230        1.79710
// ------------------------------
//
//************************************************************

int ChangeXmol(char fninp[], char fnoup[], int FDEBUG, char desc[])
{
    //******************************
    // define consts and variables
    //******************************
    FILE *fo, *fp;
    char fntmp[200];
    char *c;
    
    int nmol, igomi, ndat;
    double dgomi;
    int i, istep;
    double x, y, z;
    
    if (FDEBUG) printf ("Making xmol XYZ file...\n");
    
    //******************************
    // Reading header
    //******************************
    fp = fopen(fninp,"r");
    if (fp==NULL || ferror(fp)){
        printf("Error: file access failed for %s\n",fninp);
        return(EXIT_FAILURE);
    }
    fscanf(fp,"%d %d %d", &igomi, &nmol, &ndat);
    fscanf(fp,"%lf %lf %lf",&dgomi, &dgomi, &dgomi);
    fscanf(fp,"%lf %lf", &dgomi, &dgomi);
    if (FDEBUG) {
        printf (" -- nmol: %d\n",nmol);
        printf (" -- ndat: %d\n",ndat);
    }
    
    for(istep=1;istep<=ndat;istep++)
    {
        if (FDEBUG && istep%10==0)
            printf (" -- step:%4d\n",istep);
        
        //******************************
        // open output file
        //******************************
        strcpy(fntmp, fnoup);
        c = strrchr(fntmp, (int)'.');
        if (*c==(char)NULL) c = &(fntmp[strlen(fnoup)]);
        sprintf (c, "%03d.xyz", istep);
        fo = fopen(fntmp,"w");
        
        //******************************
        // Reading and writing header
        //******************************
        fprintf(fo,"%d\n",nmol);
        fprintf(fo,"%s\n",desc);
        
        //******************************
        // Reading and writing every scene
        //******************************
        for(i=0;i<nmol;i++){
            fscanf(fp,"%lf %lf %lf",&x,&y,&z);
            fprintf(fo,"C    ");
            fprintf(fo,"%14.6lf  ",x);
            fprintf(fo,"%14.6lf  ",y);
            fprintf(fo,"%14.6lf  ",z);
            fprintf(fo,"\n");
        }
        fclose(fo);
    }
    fclose(fp);
    return(EXIT_SUCCESS);
}

