
/************************************************************/
/****                                                    ****/
/****      wrapping - nanotube coordinate generator      ****/
/****                                                    ****/
/****        2006.2.26   written by Mototeru OBA         ****/
/****                                                    ****/
/************************************************************/
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "change.h"

const char DEFAULT_DATNAME[30] = "swnt_pos";
const char DEFAULT_OPTBRE[30] = "optbre.dat";
const char DEFAULT_OPTTB[30] = "opttb.dat";
const double PI    = 3.1415926535897932384626433832795;
const double twoPI = 6.2831853071795864769252867665590;

//************************************************************
// flag FDEBUG
// flag FCHECK
//************************************************************
// These are switches for developper.
// Normally, they should be set to zero.
//
// === FDEBUG ===
//   If FDEBUG != 0...
//     This program displays various debug information.
//   IF FDEBUG == 0...
//     This program doesn't display any debug information.
//
// === FCHECK ===
//   If FCHECK != 0...
//     This program checks the unit cell made by MakeUnit().
//     Function CheckStructure() counts the number of bonds
//     of each atom are counted and displays the result. 
//   IF FCHECK == 0...
//     This program doesn't check the unit cell.
//************************************************************
const int FDEBUG = 0;
const int FCHECK = 0;

//************************************************************
// define structs
//************************************************************
// Struct vecxy describes 2D xy coordinate.
// Struct vecxy describes 3D cylindrical coordinate.
//************************************************************
struct vecxy {
    double x;
    double y;
};

struct intpos {
    int c1;
    int c2;
    int cb;
};

struct vecgra {
    struct vecxy a1;
    struct vecxy a2;
    struct vecxy ab;
};

struct veccnt {
    int n, m;
    struct vecxy ch;
    struct vecxy t;
    struct vecxy axis;
    double rch, rt;
    double raxis;
    double diam, angle;
    int nunit, natom;
    int idr, it1, it2;
};

//************************************************************
// define functions
//************************************************************
void CheckStructure(struct intpos[]);
void MakeLattice(struct intpos[]);
void MakeUnit(int, int, struct intpos[]);
void OutDat(char[], struct intpos[], struct vecgra, struct veccnt);
void ReadToken(FILE*, char[]);
void SetParam (char[], char[]);
void SetParamArg(int, char*[]);
void SetParamFile(char fbin[], char fin[]);
int foption(char[]);
int MakeOption(char fn[], char str[], char *argv[]);
int CountAtom (struct vecgra vg, struct veccnt vs, struct intpos ipos[], double lenunit);

//------------------------------
// Functions for setting the structure.
struct vecgra SetGraLattice (double lbond, char fn[]);
struct veccnt SetCntLattice (int intn, int intm, struct vecgra vg);

//------------------------------
// Functions for displaying data
void PrintLattice(struct vecgra vg, struct veccnt vs);
void PrintOption(void);

//------------------------------
// Mathematical functions
int gcm (int ia, int ib);
double dot_product(struct vecxy v1, struct vecxy v2);
double vabs(struct vecxy vev);

//------------------------------
// Other utility functions
void filepath(char c[], char cpath[]);

//************************************************************
// define variables
//************************************************************
int cn;
int cm;
double lacc = 1.44;
char binpath[200];
char outpath[200] = "";
char optfile[200] = "";
char outfile[200] = "";
int nwrap = 0;
int iaxis = 0;
double magaxis = 1.0;
double magdiam = 1.0;
double shift[3]  = {0.0, 0.0, 0.0};
int fprint = 0;
int fcenter = 1;
enum changefile {nochange, pv, xmol, tinker};
enum changefile fchange = nochange;
double lenunit = -20.0;

//************************************************************
// Main Function
//************************************************************
int main(int argc, char *argv[])
{
    char datfile[200] = "swnt_wrapping.wrp";
    char cdesc[200];
    struct intpos *ipos;
    struct vecgra vg;
    struct veccnt vs;
    
    //------------------------------
    // Start main function.
    //------------------------------
    if (argc==1) {
        printf("Error: No arguments.\n");
        exit(EXIT_FAILURE);
    }
    if (FDEBUG) printf("Debug mode: on\n");
    if (FCHECK) printf("Check mode: on\n");
    
    //------------------------------
    // Reading parameters
    //------------------------------
    filepath(argv[0],binpath);
    strcpy(optfile,binpath);
    strcat(optfile,DEFAULT_OPTTB);
    if (argc==2) {
        filepath(argv[1],outpath);
        SetParamFile (argv[0],argv[1]);
    } else {
        SetParamArg (argc, argv);
    }
    
    //------------------------------
    // Setting parameters of graphene and SWNT.
    //------------------------------
    if (FDEBUG) {
        printf ("\n");
        printf ("Making graphene and SWNT structure.\n");
    }
    vg = SetGraLattice(lacc,optfile);
    vs = SetCntLattice(cn,cm,vg);
    if (lenunit<0.0) lenunit /= -vs.raxis;
    if (FDEBUG) PrintOption();
    
    //------------------------------
    // Display the structural parameters
    //------------------------------
    if (fprint) PrintLattice(vg,vs);
    if (nwrap<0) exit(EXIT_SUCCESS);
    
    //------------------------------
    // Making unit cell
    //------------------------------
    ipos = (struct intpos*)malloc((vs.natom+10) * sizeof(struct intpos));
    MakeUnit(cn,cm,ipos);
    if (FCHECK) CheckStructure(ipos);
    
    //------------------------------
    // Open file and output header
    //------------------------------
    OutDat(datfile,ipos,vg,vs);
    free(ipos);
    
    switch(fchange) {
    case(nochange):
        if (strcmp(outfile,"")==0) {
            strcpy(outfile,outpath);
            strcat(outfile,DEFAULT_DATNAME);
            strcat(outfile,".dat");
        }
        if (remove(outfile)!=0) {
            printf ("Warning: failed to remove old %s file.\n",outfile);
        }
        if (rename(datfile,outfile)!=0) {
            printf ("Warning: failed to save %s file.\n",outfile);
            printf (" -- Data is saved as %s.\n",datfile);
        }
        break;
    case(pv):
        ChangePV(datfile,outfile,FDEBUG,fcenter);
        remove(datfile);
        break;
    case(xmol):
        sprintf (cdesc, "(%d,%d) SWNT-XMOL XYZ FORMAT",cn,cm);
        ChangeXmol(datfile,outfile,FDEBUG,cdesc);
        remove(datfile);
        break;
    case(tinker):
        sprintf (cdesc, "(%d,%d) SWNT-TINKER XYZ FORMAT",cn,cm);
        ChangeTinker(datfile,outfile,FDEBUG,cdesc);
        remove(datfile);
        break;
    }
    //------------------------------
    // exit program
    //------------------------------
    return(EXIT_SUCCESS);
}

//************************************************************
// Function OutDat()
//************************************************************
void OutDat(char fn[], struct intpos ipos[], struct vecgra vg, struct veccnt vs)
{
    FILE *fp;
    int nmol;
    int i, j, iwrap;
    int fperiod;
    int ix = iaxis;
    int iy = (ix+1) % 3;
    int iz = (ix+2) % 3;
    struct vecxy vec;
    double cwrap;
    double qx[3],th,z;
    double vl[3], sft[3];
    double lenaxis = lenunit * vs.raxis;
    
    fperiod = (fabs(lenunit-floor(lenunit+0.5))<1.0e-8 ? 1 : 0);
    
    //------------------------------
    // Calculate cell size.
    //------------------------------
    sft[0] = shift[0];
    sft[1] = shift[1];
    sft[2] = shift[2];
    vl[ix] = fabs(magaxis) * lenaxis;
    vl[iy] = fabs(magdiam) * vs.diam * 4.0;
    vl[iz] = fabs(magdiam) * vs.diam * 4.0;
    if (fcenter) {
        sft[ix] -= vl[ix]/2.0;
    } else {
        sft[iy] += vl[iy]/2.0;
        sft[iz] += vl[iz]/2.0;
    }
    vl[0] += 2.0 * fabs(shift[0]);
    vl[1] += 2.0 * fabs(shift[1]);
    vl[2] += 2.0 * fabs(shift[2]);
    
    //------------------------------
    // Open files and write header
    //------------------------------
    nmol = CountAtom (vg, vs, ipos, lenunit);
    fp = fopen(fn,"w");
    fprintf (fp,"0 %d %d\n",nmol,nwrap+1);
    fprintf (fp,"%lf  %lf  %lf\n", vl[0], vl[1], vl[2]);
    fprintf (fp,"0.0  0.0\n");
    
    for (iwrap=0; iwrap<=nwrap; iwrap++) {
      cwrap = (nwrap==0 ? 1.0 : (double)iwrap/(double)nwrap);
      for (j=0; j<=1+(int)lenunit; j++) {
        if (fperiod && j>=(int)lenunit) continue;
        for (i=0; i<vs.natom; i++) {
            vec.x  = (double)(ipos[i].c1 + j * vs.it1) * vg.a1.x;
            vec.x += (double)(ipos[i].c2 + j * vs.it2) * vg.a2.x;
            vec.x += (double)ipos[i].cb * vg.ab.x;
            vec.y  = (double)(ipos[i].c1 + j * vs.it1) * vg.a1.y;
            vec.y += (double)(ipos[i].c2 + j * vs.it2) * vg.a2.y;
            vec.y += (double)ipos[i].cb * vg.ab.y;
            th = dot_product(vec,vs.ch) / pow(vs.rch,2);
            th -= floor(th+0.4999999);
            th *= twoPI;
            z  = dot_product(vec,vs.axis);
            if (!fperiod && (z-1.0e-8)>lenaxis) continue;
            if (fperiod) z -= floor(z/lenaxis+1.0e-8) * lenaxis;
            
            if (magaxis>0.0)
                qx[ix] = z * magaxis;
            else
                qx[ix] = (z-lenaxis) * magaxis;
            
            if (cwrap<1.0e-8) {
                qx[iy] = th*(vs.diam*magdiam)/2.0;
                qx[iz] = 0.0;
            } else {
                qx[iy] = ((vs.diam*magdiam)/2.0/cwrap) * sin(th*cwrap);
                qx[iz] = ((vs.diam*magdiam)/2.0/cwrap) * (1.0-cos(th*cwrap));
            }
            qx[iz] -= (vs.diam*magdiam)/2.0;
            
            fprintf (fp, "%14.6f  ", qx[0] + sft[0]);
            fprintf (fp, "%14.6f  ", qx[1] + sft[1]);
            fprintf (fp, "%14.6f  ", qx[2] + sft[2]);
            fprintf (fp, "\n");
        }
      }
    }
    fclose(fp);
    return;
}

int CountAtom (struct vecgra vg, struct veccnt vs, struct intpos ipos[], double lenunit)
{
    int i;
    struct vecxy vec;
    int fperiod;
    int nmol;
    double z, lenaxis;
    
    fperiod = (fabs(lenunit-floor(lenunit+0.5))<1.0e-8 ? 1 : 0);
    lenaxis = lenunit * vs.raxis;
    
    //------------------------------
    // Count the number of atoms
    //------------------------------
    nmol = vs.natom * (int)(floor(lenunit+1.0e-8)+1.0e-8);
    if (!fperiod) {
        for (i=0; i<vs.natom; i++) {
            vec.x  = (double)ipos[i].c1 * vg.a1.x;
            vec.x += (double)ipos[i].c2 * vg.a2.x;
            vec.x += (double)ipos[i].cb * vg.ab.x;
            vec.y  = (double)ipos[i].c1 * vg.a1.y;
            vec.y += (double)ipos[i].c2 * vg.a2.y;
            vec.y += (double)ipos[i].cb * vg.ab.y;
            z  = floor(lenunit+1.0e-8) * vs.raxis + dot_product(vec,vs.axis);
            if ((z-1.0e-8)>lenaxis) continue;
            nmol++; 
        }
    }
    return(nmol);
}

//************************************************************
// Function SetParamArg()
//************************************************************
void SetParamArg(int argc, char *argv[])
{
    int i;
    if (FDEBUG) {
        printf("\n");
        printf("Set parameters.\n");
    }
    
    cn = atoi(argv[1]);
    cm = atoi(argv[2]);
    if (cn==0 || cn<cm) {
        printf("Error: incorrect chirality=(%d,%d).\n",cn,cm);
        exit(EXIT_FAILURE);
    }
    if (FDEBUG) printf(" -- chirality = (%d,%d).\n", cn, cm);
    i=3;
    while (i<argc) {
        if (!foption(argv[i])) {
            printf("Error: incorrect arguments: <%s>\n",argv[i]);
            exit(EXIT_FAILURE);
        }
        if (i+1==argc || foption(argv[i+1])) {
            SetParam(argv[i],""); i++;
        } else {
            SetParam(argv[i],argv[i+1]); i+=2;
        }
    }
}

//************************************************************
// Function SetParamFile()
//************************************************************
void SetParamFile(char fbin[], char fin[])
{
    char sopt[10000];
    int argc;
    char *argv[100];
    
    if (FDEBUG) {
        printf("\n");
        printf("Read parameters from file <%s>.\n", fin);
    }
    argv[0] = &(fbin[0]);
    argc = MakeOption(fin, sopt, argv);
    SetParamArg (argc, argv);
    return;
}

//************************************************************
// Function MakeOption()
//************************************************************
int MakeOption(char fn[], char str[], char *argv[])
{
    int i;
    int iarg, ichr;
    char sgomi[200];
    FILE *fp;
    
    fp = fopen(fn,"r");
    if (fp==NULL || ferror(fp)) {
        printf("Error: file access failed for <%s>\n",fn);
        return(-1);
    }
    ichr = 0;
    iarg = 1;
    argv[iarg] = &(str[0]);
    while (feof(fp)==0) {
        str[ichr] = (char)fgetc(fp);
        if (str[ichr]<0) {fgetc(fp); continue;}
        switch (str[ichr]) {
        case('"'):
            if (ichr>0 && str[ichr-1]!='\0') {
                str[ichr++]='\0';
                argv[++iarg]=&(str[ichr]);
            }
            while((str[ichr]=(char)fgetc(fp))!='"') ichr++;
            str[ichr++]='\0';
            argv[iarg++]=&(str[ichr]);
            break;
        case(','):
        case(' '):
        case('\n'):
            if (ichr>0 && str[ichr-1]!='\0') {
                str[ichr++]='\0';
                argv[++iarg] = &(str[ichr]);
            }
            break;
        case('#'): fgets(sgomi,200,fp); break;
        default: ichr++; break;
        }
    }
    if (ichr==0 || str[ichr-1]!='\0') {
        str[ichr] = '\0';
        iarg++;
    } else {
        argv[iarg] = NULL;
    }
    if (FDEBUG)
        for (i=0; i<iarg; i++) printf (" -- Token: <%s>\n",argv[i]);
    return(iarg);
}

//************************************************************
// Function foption()
//************************************************************
int foption(char arg[])
{
    if ('0'<=arg[1] && arg[1]<='9') return(0);
    if (arg[0]!='/' && arg[0]!='-') return(0);
    return(1);
}

//**********************************************************
// Function SetParam()
//**********************************************************
// This function sets the options to global variables.
// Variable arg1[] is the type of option and arg2[] is
// the value of the option.
//**********************************************************
void SetParam (char arg1[], char arg2[])
{
    int ierr = 0;
    char cp[3], *c;

    cp[2] = '\0';
    if (strlen(arg1)<=3) {
        cp[0] = arg1[1];
        cp[1] = arg1[2];
    } else {
        cp[0] = arg1[1];
        c = strpbrk(arg1,":");
        if (c==(char *)NULL) cp[1]='\0'; else cp[1]=*(c+1);
    }
    
    if ('A'<=cp[0] && cp[0]<='Z') cp[0] += 'a' - 'A';
    if ('A'<=cp[1] && cp[1]<='Z') cp[1] += 'a' - 'A';
    if (FDEBUG!=0) printf(" -- set Param: <-%s>, <%s>\n", cp, arg2);
    
    switch (cp[0]) {
    case('a'):
        switch(cp[1]) {
        case('x'): iaxis = 0; break;
        case('y'): iaxis = 1; break;
        case('z'): iaxis = 2; break;
        default: ierr = 2; break;
        } break;
    case('b'):
        lacc = -1.0;
        strcpy(optfile,binpath);
        strcat(optfile,DEFAULT_OPTBRE);
        break;
    case('c'): lacc = (strcmp(arg2,"")!=0 ? atof(arg2) : 1.44 ); break;
    case('d'): fprint = 1; break;
    case('f'): strcpy(outfile,arg2); ierr=1; break;
    case('g'):
        ierr = 1;
        lacc = -1.0;
        if (strcmp(arg2,"")!=0) strcpy(optfile,arg2);
        break;
    case('l'): lenunit=-atof(arg2); ierr=1; break;
    case('p'):
        fchange = pv;
        if (strcmp(outfile,"")==0) {
            strcpy(outfile,outpath);
            strcat(outfile,DEFAULT_DATNAME);
            strcat(outfile,".pv");
        } break;
    case('r'):
        switch(cp[1]) {
        case('d'): magdiam=atof(arg2); break;
        case('a'): magaxis=atof(arg2); break;
        default: ierr=2; break;
        } break;
    case('s'):
        switch(cp[1]) {
        case('x'): shift[0] = atof(arg2); break;
        case('y'): shift[1] = atof(arg2); break;
        case('z'): shift[2] = atof(arg2); break;
        default: ierr = 2; break;
        } break;
    case('t'):
        lacc = -1.0;
        strcpy(optfile,binpath);
        strcat(optfile,DEFAULT_OPTTB);
		break;
    case('u'): lenunit = fabs(atof(arg2)); ierr=1; break;
    case('w'): nwrap = (strcmp(arg2,"")==0 ? 20 : atoi(arg2)); break;
    case('x'):
        switch(cp[1]) {
        case('t'): fchange = tinker; break;
        case('x'): fchange = xmol;   break;
        default: ierr = 2; break;
        }
        if (strcmp(outfile,"")==0) {
            strcpy(outfile,outpath);
            strcpy(outfile,DEFAULT_DATNAME);
            strcat(outfile,".xyz");
        }
        break;
    case('z'):
        switch(cp[1]) {
        case('c'): fcenter = 0; break;
        case('m'): fcenter = 1; break;
        default: ierr = 2; break;
        } break;
    default: ierr = 2; break;
    }
    if (ierr==1) ierr = (strcmp(arg2,"")==0 ? 1 : 0);
    switch (ierr) {
        case(1): printf ("Error: no argument (option %s).\n",arg1); break;
        case(2): printf ("Error: Invalid argument (option %s).\n",arg1); break;
    }
    if (ierr) exit(EXIT_FAILURE);
}


//************************************************************
// Function CheckStructure()
//************************************************************
// This function checks whether the number of bonds in a atom
// is three or not. If an atom doesn't have three bonds,
// warning message is displayed.
//************************************************************
void CheckStructure(struct intpos ipos[])
{
    const double RCUT = 1.8;
    int *nbond;
    int i1, i2, iflag;
    double th ,z, rabs2;
    double x1, y1, z1;
    double x2, y2, z2;
    struct vecxy vec;
    struct vecgra vg;
    struct veccnt vs;
    
    if (FDEBUG) {
        printf ("\n");
        printf ("Checking the number of bonds in each atom...\n");
    }
    
    //******************************
    // Set cylindrical structure
    //******************************
    vg = SetGraLattice(1.44,NULL);
    vs = SetCntLattice(cn,cm,vg);
    
    //******************************
    // Allocate memory and set initial value.
    //******************************
    nbond = (int *) malloc (vs.natom * sizeof(int));
    for (i1=0; i1<vs.natom; i1++) {
        nbond[i1]=0;
    }
    
    //******************************
    // Calculate bond length of all atom pairs.
    //******************************
    for (i1=0; i1<vs.natom; i1++) {
        //------------------------------
        // Make xyz-coordinate of atom i1.
        vec.x  = (double)ipos[i1].c1 * vg.a1.x;
        vec.x += (double)ipos[i1].c2 * vg.a2.x;
        vec.x += (double)ipos[i1].cb * vg.ab.x;
        vec.y  = (double)ipos[i1].c1 * vg.a1.y;
        vec.y += (double)ipos[i1].c2 * vg.a2.y;
        vec.y += (double)ipos[i1].cb * vg.ab.y;
        th = dot_product(vec,vs.ch) / pow (vs.rch, 2) * twoPI;
        x1 = vs.diam * cos(th) / 2.0;
        y1 = vs.diam * sin(th) / 2.0;
        z1  = dot_product(vec,vs.t)  / vs.rt;
        for (i2=i1+1; i2<vs.natom; i2++) {
            //------------------------------
            // Make xyz-coordinate of atom i2.
            vec.x  = (double)ipos[i2].c1 * vg.a1.x;
            vec.x += (double)ipos[i2].c2 * vg.a2.x;
            vec.x += (double)ipos[i2].cb * vg.ab.x;
            vec.y  = (double)ipos[i2].c1 * vg.a1.y;
            vec.y += (double)ipos[i2].c2 * vg.a2.y;
            vec.y += (double)ipos[i2].cb * vg.ab.y;
            th = dot_product(vec,vs.ch) / pow (vs.rch, 2) * twoPI;
            x2 = vs.diam * cos(th) / 2.0;
            y2 = vs.diam * sin(th) / 2.0;
            z2  = dot_product(vec,vs.t)  / vs.rt;
            
            //------------------------------
            // Calculate the length of i1 and i2.
            z = z1 - z2;
            z  -= floor(z/vs.raxis+0.5)*vs.raxis;
            rabs2 = pow(x1-x2,2) + pow(y1-y2,2) + pow(z,2);
            
            //------------------------------
            // Check whether the bond exists.
            if (rabs2<pow(RCUT,2)) {nbond[i1]++; nbond[i2]++;}
        }
    }
    
    //******************************
    // Display the result
    //******************************
    iflag = 1;
    for (i1=0; i1<vs.natom; i1++) {
        if (nbond[i1]==3) continue;
        iflag = 0;
        printf ("Error: Atom %4d: %d bonds\n", i1, nbond[i1]);
        exit(EXIT_FAILURE);
    }
    if (iflag) printf ("OK: All atoms has 3 bonds.\n");
    
    //******************************
    // Deallocate memory and exit
    //******************************
    free(nbond);
    return;
}

//************************************************************
// void MakeUnit()
//************************************************************
void MakeUnit(int cn, int cm, struct intpos ipos[])
{
    int iatom = 0;
    int i1, i2, ib;
    struct vecxy vec, va;
    struct vecgra vg;
    struct veccnt vs;
    double th ,z;
    
    if (FDEBUG) {
        printf ("\n");
        printf ("Making unit cell...\n");
    }
    //------------------------------
    // Set cylindrical structure
    //------------------------------
    vg = SetGraLattice(1.44,(char *)NULL);
    vs = SetCntLattice(cn,cm,vg);
    
    for (i1=0; i1<=vs.it1+vs.n; i1++) {
    for (i2=vs.it2; i2<=vs.m; i2++) {
        va.x = (double)i1 * vg.a1.x + (double)i2 * vg.a2.x;
        va.y = (double)i1 * vg.a1.y + (double)i2 * vg.a2.y;
        for (ib=0; ib<=1; ib++) {
            vec.x = va.x + (double)ib * vg.ab.x;
            vec.y = va.y + (double)ib * vg.ab.y;
            th = dot_product(vec,vs.ch) / pow (vs.rch, 2);
            z  = dot_product(vec,vs.t)  / (vs.rt * vs.raxis);
            if (th<-0.0001 || th>0.9999) continue;
            if (z <-0.0001 || z> 0.9999) continue;
            ipos[iatom].c1 = i1;
            ipos[iatom].c2 = i2;
            ipos[iatom].cb = ib;
            iatom++;
        }
    }
    }
    return;
}

//**********************************************************
// Functions for setting the structure
//**********************************************************
// SetGraLattice()
//     sets the structural vectors of 2D graphene.
//     Parameters are the member of struct vecgra:
//         a1, a2 ... the primitive lattice vectors,
//         ab ... the vector from A atom to B atom.
//     If lbond > 0, parameters are set to the cylindrical
//     structure with bond length lacc, and if lbond <= 0,
//     parameters are read from file fn[].
//
// SetCntLattice()
//     sets the structural vectors and parameters of SWNT.
//     The parameters are set using chiral indices (intn,intm)
//     and the structural vectors of graphene vg.
//**********************************************************
struct vecgra SetGraLattice (double lbond, char fn[])
{
    FILE *fp;
    struct vecgra vg;
    int i1, i2, igomi;
    double dgomi;
    
    if (lbond>0) {
        if (FDEBUG) {
            printf (" -- set cylindrical structure ");
            printf ("(bond length = %lf).\n",lbond);
        }
        vg.a1.x = lbond * (3.0/2.0);
        vg.a1.y = lbond * (sqrt(3.0)/2.0);
        vg.a2.x =  vg.a1.x;
        vg.a2.y = -vg.a1.y;
        vg.ab.x = lbond;
        vg.ab.y = 0.0;
    } else {
        if (FDEBUG) {
            printf (" -- read structure file from <%s>.\n",fn);
        }
        fp = fopen(fn,"r");
        if (fp==NULL || ferror(fp)){
            printf("Error: file access failed for %s\n",fn);
            exit(EXIT_FAILURE);
        }
        while(fgetc(fp)!='\n'); //skip one line
        i1 = 0;
        i2 = 0;
        while (i1!=cn || i2!=cm) {
            fscanf(fp,"%d %d", &i1, &i2);
            fscanf(fp,"%lf %lf %d", &dgomi, &dgomi, &igomi);
            fscanf(fp,"%lf %lf", &(vg.a1.x), &(vg.a1.y));
            fscanf(fp,"%lf %lf", &(vg.a2.x), &(vg.a2.y));
            fscanf(fp,"%lf %lf", &(vg.ab.x), &(vg.ab.y));
            if (feof(fp)!=0) {
                printf ("Error: parameters for (%d,%d) SWNT ",cn,cm);
                printf ("do not exist in the file.\n");
                fclose(fp);
                exit(EXIT_FAILURE);
            }
        }
        fclose(fp);
    }
    return(vg);
}

//******************************
struct veccnt SetCntLattice (int intn, int intm, struct vecgra vg)
{
    struct veccnt vs;
    
    if (FDEBUG) {
        printf (" -- set structure of ");
        printf ("(%d,%d) SWNT.\n",intn, intm);
    }
        
    //------------------------------
    // chirality(n,m)
    //------------------------------
    vs.n = intn;
    vs.m = intm;
    
    //------------------------------
    // chiral vector(ch) and its length(rch)
    //------------------------------
    vs.ch.x = (double)intn * vg.a1.x + (double)intm * vg.a2.x;
    vs.ch.y = (double)intn * vg.a1.y + (double)intm * vg.a2.y;
    vs.rch = vabs(vs.ch);
    
    //------------------------------
    // diameter(diam) and chiral angle(angle)
    //------------------------------
    vs.diam = vs.rch / PI;
    vs.angle = dot_product(vg.a1, vs.ch) / vabs(vg.a1) / vabs(vs.ch);
    if (vs.angle>1.0) vs.angle = 1.0 - 1.0e-16;
    vs.angle = acos(vs.angle) * 180.0 / PI;
    
    //------------------------------
    // translation vector(t) and its length(rt)
    //------------------------------
    vs.idr = gcm (2*intm+intn, 2*intn+intm);
    vs.it1 =  (2*intm+intn) / vs.idr;
    vs.it2 = -(2*intn+intm) / vs.idr;
    vs.t.x = (double)vs.it1 * vg.a1.x + (double)vs.it2 * vg.a2.x;
    vs.t.y = (double)vs.it1 * vg.a1.y + (double)vs.it2 * vg.a2.y;
    vs.rt = vabs(vs.t);
    
    
    //------------------------------
    // axis direction(axis) and its length(raxis)
    //------------------------------
    vs.axis.x = -vs.ch.y / vs.rch;
    vs.axis.y =  vs.ch.x / vs.rch;
    vs.raxis  = dot_product(vs.t,vs.axis);
    if (vs.raxis<0.0) {
        vs.axis.x *= -1.0;
        vs.axis.y *= -1.0;
        vs.raxis  *= -1.0;
    }
    
    //------------------------------
    // number of graphene unit cells in a SWNT unit cell
    //------------------------------
    vs.nunit = 2 * (intn*intn + intn*intm + intm*intm) / vs.idr;
    vs.natom = 2 * vs.nunit;
    return(vs);
}

//**********************************************************
// Functions for displaying data
//**********************************************************
// PrintLattice()
//     displays the parameters of graphene and SWNT. 
//
// PrintOption()
//     displays the value of global variables. 
//**********************************************************
void PrintLattice(struct vecgra vg, struct veccnt vs)
{
    //------------------------------
    // display data of (cn,cm) SWNT
    //------------------------------
    printf("\n");
    printf("Structural parameters of (%d, %d) SWNT\n", vs.n, vs.m);
    
    //------------------------------
    printf("Basic parameters of SWNT\n");
    printf(" -- diameter:     %lf[AA]\n",  vs.diam);
    printf(" -- chiral angle: %lf[deg]\n", vs.angle);
    printf(" -- number of atoms in SWNT unit cell: %d\n", vs.natom);
    
    //------------------------------
    printf("Primitive lattice vectors of graphene\n");
    printf(" -- va1[AA] = (%lf, %lf)\n", vg.a1.x, vg.a1.y);
    printf(" -- va2[AA] = (%lf, %lf)\n", vg.a2.x, vg.a2.y);
    printf(" -- vab[AA] = (%lf, %lf)\n", vg.ab.x, vg.ab.y);
    
    //------------------------------
    printf("Primitive lattice vectors of SWNT on 2D graphene\n");
    printf(" -- Ch[AA] = %d va1 %+d va2 = (%lf, %lf)\n",
                 vs.n, vs.m, vs.ch.x, vs.ch.y);
    printf(" -- T[AA]  = %d va1 %+d va2 = (%lf, %lf)\n",
                vs.it1, vs.it2, vs.t.x,  vs.t.y);
    return;
}

//******************************
void PrintOption(void)
{
    printf ("\n");
    printf ("Options are:\n");
    printf (" -- (cn,cm) = (%d,%d)\n", cn, cm);
    printf (" -- lacc    = %lf\n", lacc);
    printf (" -- optfile = %s\n", optfile);
    printf (" -- outfile = %s\n", outfile);
    printf (" -- nwrap   = %d\n", nwrap);
    printf (" -- iaxis   = %d\n", iaxis);
    printf (" -- magaxis = %lf\n", magaxis);
    printf (" -- magdiam = %lf\n", magdiam);
    printf (" -- shiftx  = %lf\n", shift[0]);
    printf (" -- shifty  = %lf\n", shift[1]);
    printf (" -- shiftz  = %lf\n", shift[2]);
    printf (" -- lenunit = %lf\n", lenunit);
    printf (" -- fprint  = %d\n", fprint);
    printf (" -- fcenter = %d\n", fcenter);
    printf (" -- fchange = %d\n", fchange);
    return;
}

//************************************************************
// Mathematical functions and subroutines
//************************************************************
// int gcm (int ia, int ib)
//   Calculating the greatest common measure of two indices.
//
// double dot_product(struct vecxy v1, struct vecxy v2)
//   Calculating dot_product of two 2D vectors.
//
// double vabs(struct vecxy v)
//   Calculating the absolute value of a 2D vector.
//************************************************************
int gcm (int ia, int ib)
{
    int i;
    if (ia<0) ia = -ia;
    if (ib<0) ib = -ib;
    if (ia>ib) {i=ia; ia=ib; ib=i;} // swap ia and ib
    if (ia==0) return(ib);
    while(-1){
        ib = ib % ia;
        if (ib==0) return(ia);
        i=ia; ia=ib; ib=i; // swap ia and ib
    }
}

//******************************
double dot_product(struct vecxy v1, struct vecxy v2)
{
    double dret;
    dret = v1.x * v2.x + v1.y * v2.y;
    return(dret);
}

//******************************
double vabs(struct vecxy vec)
{
    double dret;
    dret = sqrt(vec.x * vec.x + vec.y * vec.y);
    return(dret);
}

//************************************************************
// Function filepath()
//************************************************************
void filepath(char c[], char cpath[])
{
    char *cp;
    strcpy(cpath,c);
    cp = strrchr(cpath, (int)'/');
    if (cp==NULL) cp = strrchr(cpath, (int)'\\');
    if (cp!=NULL) {cp++; *cp='\0';} else strcpy(cpath,"");
}
